 ; Ŀ
 ;   Fang - run Lisp routines on a directory of drawings.                  
 ;   Fang now requires no input except for lisp names and uses all files   
 ;   in the current directory.                                             
 ;   The variant Fangs offers to do files in subdirectories, can extract   
 ;   filenames from blocks in an index drawing, and lets you edit the      
 ;   list of drawings before writing the batch.                            
 ;   Copyright 1991 - 2009 by Rocket Software Ltd.                         
 ;   Unless you like to do tedious work while your computer has coffee.    
 ;   And eats rats.                                                        
 ;                                                                         
 ;   Notes:                                                                
 ;   Don't forget to add the following line to Acad.pgp:                   
 ;   not, notepad.exe, 8,Notepad: ,                                        
 ;                                                                         
 ;   Fang now writes an error file, Error.txt, containing the names        
 ;   of any files which were unavailable or locked.                        
 ;                                                                         
 ;   Premade filename list files can contain comments either after         
 ;   the drawing name or on separate lines; each must be preceded by a     
 ;   semicolon and (if it is on the same line as a filename) separated     
 ;   from the filename by a space.                                         
 ;   Empty lines are ok, a leading space will cause any line to be         
 ;   ignored.                                                              
 ;                                                                         
 ;   Fang can now use, in addition to its standard data files, .csv files  
 ;   made from an ordered titleblock database file.  This allows one to    
 ;   use a database of drawing names, titles, etc. to plot a specific      
 ;   set of files in a desired order.                                      
 ;   The first field on each line must be the drawing filename.            
 ;   Any file which doesn't have a path will be assumed to be in the       
 ;   current directory.                                                    
 ;   Lines can be commented out by prefacing them with a semicolon.        
 ;   Later: This version can extract file names from index blocks in a     
 ;   file list drawing.  It uses the file name Files.dat, and assumes      
 ;   that any drawings in the list are in the current directory.           
 ;                                                                         
 ; 

 ; Ŀ
 ;   Subroutine Awp: find the width of an attribute at a given position    
 ;   in a block insertion.                                                 
 ;   Arguments: Enam, an insert ename.                                     
 ;              Poz, a zero based position in the block.                   
 ;   Returns a width, possibly zero, nil if there was no attribute         
 ;   at that position.                                                     
 ; 
 (DEFUN AWP (enam poz / num entt wida stop)
  (setq num 0)
  (while (and (null stop) (setq enam (entnext enam)))
         (cond ((= (cdr (assoc 0 (setq entt (entget enam)))) "SEQEND")
                (setq wida ())
                (setq stop t))
               ((= num poz)
                (setq wida (twid entt))
                (setq stop t)))
         (setq num (1+ num)))
 wida)
 ; Ŀ
 ;   Subroutine Awp end.                                                   
 ; 

 ; Ŀ
 ;   CeeCee - make a csv file into a directory list, assuming that the     
 ;   first field in each line is a drawing name.  (Check for this?)        
 ;   Arguments: Filnam, a file name.                                       
 ;   Returns a list of file names.                                         
 ;   Calls Splat and Izla.                                                 
 ; 
 (DEFUN CEECEE (filnam / fn fnam txtstr flist noam prefa)
 ; Ŀ
 ;   Open the file, read and process each line.                            
 ; 
  (setq fn (open filnam "r"))
  (while (setq txtstr (read-line fn))
 ; Ŀ
 ;   Call Splat to chop the string up at commas, save the first field      
 ;   from any good (not empty or comment) lines to a list.                 
 ; 
         (setq flist (splat "," txtstr))
         (setq txtstr (car flist))
         (if (and (/= txtstr "") (/= (substr txtstr 1 1) ";"))
             (setq noam (cons txtstr noam))))
  (close fn)
 ; Ŀ
 ;   Add the path to each drawing name if it isn't already present.        
 ; 
  (setq prefa (getvar "dwgprefix"))
  (setq noam (mapcar '(lambda (fnam)
                       (if (null (izla fnam)) (strcat prefa fnam)))
                     (reverse noam)))
 noam)
 ; Ŀ
 ;   CeeCee end.                                                           
 ; 

 ; Ŀ
 ;   Chug - string substitution engine.  Takes the search string, the      
 ;   replacement string, and the target string as arguments, and returns   
 ;   a list of the (possibly modified) target string and the number of     
 ;   changes made.                                                         
 ; 
 (DEFUN CHUG (oldstr newstr exstr / pos chnum changd newlen chunk)
  (setq pos 1)
  (setq chnum 0)
  (setq changd ())
  (setq newlen (strlen newstr))
  (setq oldlen (strlen oldstr))
  (while (= oldlen (strlen (setq chunk (substr exstr pos oldlen))))
         (if (= chunk oldstr)
             (progn
                  (setq exstr (strcat (substr exstr 1 (1- pos))
                                       newstr
                                      (substr exstr (+ pos oldlen))))
                  (setq changd t)
                  (setq chnum (1+ chnum))
                  (setq pos (+ pos newlen)))
             (setq pos (1+ pos))))
 (list exstr chnum))
 ; Ŀ
 ;   Chug end.                                                             
 ; 

 ; Ŀ
 ;   Subroutine Collie: sort entities into columns.          2007.10.02    
 ;   Arguments: ss, an ss of entities to columnise.                        
 ;              Slop, a maximum acceptable slop distance.                  
 ;                                                                         
 ;   Calls: Horiz/Lowest, Vtol/Spit, Longa/Awp/Twid, and Liss.             
 ;   Returns a list of lists: each sublist contains all the enames in a    
 ;   column in vertical order, preceded by the average column x position.  
 ; 
 (DEFUN COLLIE (ss slop / num enam xpos xlist xa totalx numofx sub column
                                                                      malist)
 ; Ŀ
 ;   Make the ((x_insert ename) ...) list.                                 
 ; 
  (setq num 0)
  (while (setq enam (ssname ss num))
         (setq num (1+ num))
         (setq xpos (cadr (assoc 10 (entget enam))))
         (setq xlist (cons (list xpos enam) xlist)))
 ; Ŀ
 ;   Order the list Xlist by first value.                                  
 ; 
  (setq xlist (horiz xlist))
 ; Ŀ
 ;   Columnise:                                                            
 ;   Add to the first sublist until reach the cutoff difference in X pos,  
 ;   then start a new sublist, continue until run out of sublists.         
 ; 
  (setq xa (caar xlist))
  (setq num 0)
  (setq totalx 0)
  (setq numofx 0)
  (while (setq sub (nth num xlist))
         (setq num (1+ num))
         (setq xpos (car sub))
 ; Ŀ
 ;   If the x position is still within the slop distance of the first      
 ;   one in the column, add the ename to the current column sublist.       
 ; 
         (if (<= (abs (- xpos xa)) slop)
             (progn
                  (setq numofx (1+ numofx))
                  (setq totalx (+ totalx xpos))
                  (setq column (cons (cadr sub) column)))
 ; Ŀ
 ;   If the x position is outside the slop distance then start a new       
 ;   column sublist.                                                       
 ; 
             (progn
                  (setq column (vtol (liss column) "y" t))
                  (setq column (cons (/ totalx numofx) column))
                  (setq malist (append malist (list column)))
                  (setq totalx xpos)
                  (setq numofx 1)
                  (setq xa xpos)
                  (setq column (list (cadr sub))))))
 ; Ŀ
 ;   There last column won't by definition set off the distance checker,   
 ;   so add it to malist.                                                  
 ; 
  (setq column (vtol (liss column) "y" t))
  (setq column (cons (/ totalx numofx) column))
  (setq malist (append malist (list column)))
 ; Ŀ
 ;   Return the list of lists.                                             
 ; 
 malist)
 ; Ŀ
 ;   Subroutine Collie end.                                                
 ; 

 ; Ŀ
 ;   Colm: make columns of blocks into a list of lists of enames,          
 ;   each sublist representing a column, then write the lot out to a       
 ;   files.dat file in the current directory.                              
 ;   Takes no arguments, calls a whole tree of stuff, returns nothing.     
 ; 
 (DEFUN COLM (/ blnam ss widtha cols gnu enam nama namlis)
  (setvar "cmdecho" 0)
 ; Ŀ
 ;   Get a block name by selection.                                        
 ; 
  (if (setq blnam (entsel "Block: "))
      (setq blnam (cdr (assoc 2 (entget (car blnam))))))
 ; (setq blnam "index")
 ; Ŀ
 ;   Get all insertions of that block.                                     
 ; 
  (setq ss (ssget "X" (list (cons 2 blnam))))
 ; Ŀ
 ;   Get the width of the longest first attribute.                         
 ; 
  (setq widtha (longa ss 0))
 ; Ŀ
 ;   Split the list into a list of lists of entities by columns.           
 ; 
  (setq cols (collie ss widtha))
 ; Ŀ
 ;   Make the column lists back into one overall list.                     
 ;   First ditch the leading column x co-ordinate.                         
 ; 
  (while (setq sub (car cols))
         (setq cols (cdr cols))
         (setq sub (cdr sub))
         (setq gnu (append gnu (list sub))))
  (setq cols (delis gnu))
 ; Ŀ
 ;   Gratuitously mark the flow of blocks.                                 
 ; 
  (larry cols)
 ; Ŀ
 ;   Make a list of file names from the list of blocks.                    
 ; 
  (setq num 0)
  (while (setq enam (nth num cols))
         (setq num (1+ num))
         (setq nama (cdr (assoc 1 (entget (entnext enam)))))
         (setq namlis (cons nama namlis)))
  (setq namlis (reverse namlis))
 ; Ŀ
 ;   Write the name list to a file.                                        
 ; 
  (litof (strcat (getvar "dwgprefix") "files.dat") namlis)
 ; Ŀ
 ;   Start Fang, bypassing the file list maker.                            
 ; 
 (princ))
 ; Ŀ
 ;   Subroutine Colm end.                                                  
 ; 

 ; Ŀ
 ;   Deep: filename file Preprocessor & valid line counter.                
 ;   Takes ona argument, the names and comments file name.                 
 ;   Calls Splat.                                                          
 ;   Returns the names only file list.                                     
 ; 
 (DEFUN DEEP (filnam / fn txtstr filnam flist noam)
  (setq fn (open filnam "r"))
  (while (setq txtstr (read-line fn))
         (while (= (substr txtstr 1 1) " ")
                (setq txtstr (substr txtstr 2)))
 ; Ŀ
 ;   Call Splat to chop the string up at semicolons and then ditch         
 ;   the second string if there is more than one, and the whole line if    
 ;   it begins with a semicolon.                                           
 ; 
         (if (and (/= txtstr "")
                  (/= (substr txtstr 1 1) ";"))
             (progn
                  (setq flist (splat ";" txtstr))
                  (setq txtstr (car flist))
                  (if (/= txtstr "") (setq noam (cons txtstr noam))))))
  (close fn)
 (reverse noam))
 ; Ŀ
 ;   Deep end.                                                             
 ; 

 ; Ŀ
 ;   Delis - make a list of lists of lists ... into a single list.         
 ;   Arguments: Alist, a list.                                             
 ;   Calls only itself (recursive.)                                        
 ;   Returns a list.                                                       
 ; 
 (DEFUN DELIS (alist / sub malist)
  (while (setq sub (car alist))
         (setq alist (cdr alist))
         (if (= (type (car sub)) 'LIST)
             (setq malist (append malist (delis sub)))
             (setq malist (append malist sub))))
 malist)
 ; Ŀ
 ;   Delis end.                                                            
 ; 

 ; Ŀ
 ;   Dirge - get a list of filenames.                                      
 ;   Arguments: Defa, nil = use defaults for path and file pattern,        
 ;                    T = ask for everything.                              
 ;   Returns a list: the path, and a list of filenames with paths.         
 ; 
 (DEFUN DIRGE (defa / pat prefa pref fils num fnam nulis)
 ; Ŀ
 ;   Set the drawing pattern.                                              
 ; 
  (if (or (null defa)
          (= "" (setq pat (getstring "Pattern <*.dwg>: "))))
      (setq pat "*.dwg"))
 ; Ŀ
 ;   Get the drawing path.                                                 
 ; 
  (setq prefa (getvar "dwgprefix"))
  (if (or (null defa)
          (= "" (setq pref (getstring (strcat "Directory <" prefa ">: ")))))
      (setq pref prefa))
 ; Ŀ
 ;   Get a list of matching files.                                         
 ; 
  (setq fils (vl-directory-files pref pat 1))
 ; Ŀ
 ;   Put the filename list in alphabetical order.                          
 ; 
  (if fils (setq fils (acad_strlsort fils)))
  (setq num 0)
  (while (and fils (setq fnam (nth num fils)))
         (setq num (1+ num))
         (setq fnam (strcat pref fnam))
         (setq nulis (cons fnam nulis)))
 (list prefa (reverse nulis)))
 ; Ŀ
 ;   Dirge end.                                                            
 ; 

 ; Ŀ
 ;   Goto - grdraw an arrow.                                               
 ;   Arguments: aa, the point to which the arrow points.                   
 ;              bb, the other end of the arrow                             
 ;              colo, the arrow colour.                                    
 ;              hi, highlight the arrow if /= 0.                           
 ;   Notes:                                                                
 ;   1. any negative colour is equivalent to xor colour - dashed white -   
 ;      which erases itself on overwrite.                                  
 ;   2. 0 erases whatever is under it, and is undocumented.                
 ;   3. Highlighting must be turned on in the first grdraw call in the     
 ;      routine, and can't be turned off in the routine.                   
 ;   These may just be anomalies in this video system.                     
 ; 
 (DEFUN GOTO (aa bb colo hi / rad basic dist bhasic ang pa pb1 pb2)
  (setq rad (/ (getvar "viewsize") 20))
  (setq basic (/ (setq dist (distance aa bb)) 4))
  (if (> basic rad) (setq basic rad))
  (if (> basic (* dist 0.75)) (setq basic (* dist 0.75)))
  (setq bhasic (/ basic 2.25))
  (setq ang (angle aa bb))
  (setq pa (polar aa ang basic))
  (setq pb1 (polar pa (+ ang (/ pi 2)) bhasic))
  (setq pb2 (polar pa (+ ang (* pi 1.5)) bhasic))
  (grdraw aa pb1 colo hi) ; hilight must be in first call, doesn't turn off
  (grdraw pb1 pb2 colo)
  (grdraw aa pb2 colo)
  (grdraw bb pa colo)
 ; (grdraw aa pa 0) ; erase line within arrowhead
 (princ))
 ; Ŀ
 ;   Goto end.                                                             
 ; 

 ; Ŀ
 ;   Subroutine Horiz - put a list in order by the first number in each    
 ;   sublist.  Takes one argument, a list, which it returns in order       
 ;   from smallest to largest first element.                               
 ;   Rehashed 2007.10.02: the first number may not be a string.            
 ; 
 (DEFUN HORIZ (nexlst / low lowp nxtsub hrzlst newlst orderd)
  (while nexlst
        (setq low (lowest nexlst))                 ; lowest leading number
        (while (and nexlst (setq nxtsub (nth 0 nexlst)))
               (setq lowp (car nxtsub))
               (if (= 'str (type lowp)) (setq lowp (read lowp)))
               (if (equal low lowp)
                   (setq hrzlst (append hrzlst (list nxtsub)))
                   (setq newlst (append newlst (list nxtsub))))
               (setq nexlst (cdr nexlst)))          ; remove 1st ent from list
        (setq orderd (append orderd hrzlst))        ; add lev sublst to levels
        (setq hrzlst ())                            ; set to () for next loop
        (setq nexlst newlst)                        ; nexlst reconstituted
        (setq newlst ()))                           ; empty new list & reuse
  orderd)
 ; Ŀ
 ;   Horiz end.                                                            
 ; 

 ; Ŀ
 ;   Izla - see if a string contains a slash or backslash, in which case   
 ;   it probably includes a path if it is a filename.                      
 ;   Arguments: Stra, a string.                                            
 ;   Returns T if a slash was found, nil otherwise.                        
 ; 
 (DEFUN IZLA (stra / num prezz sub)
  (setq num 1)
  (while (and (null prezz)
              (/= (setq sub (substr stra num 1)) ""))
         (setq num (1+ num))
         (if (member sub '("/" "\\"))
             (setq prezz T)))
 prezz)
 ; Ŀ
 ;   Izla end.                                                             
 ; 

 ; Ŀ
 ;   Subroutine Larry: draw arrows between entities in a list.             
 ;   Arguments: Liz, a list of enames.                                     
 ;   Returns nothing, calls goto.                                          
 ; 
 (DEFUN LARRY (liz / num enam prenam)
  (setq num 0)
  (while (setq enam (nth num liz))
         (setq num (1+ num))
         (if prenam (goto (cdr (assoc 10 (entget enam)))
                          (cdr (assoc 10 (entget prenam))) 3 0))
         (setq prenam enam))
 princ)
 ; Ŀ
 ;   Subroutine Larry end.                                                 
 ; 

 ; Ŀ
 ;   Lime - get a list of files in a directory and its subdirectories.     
 ;   Arguments: Cdir, the directory in which to begin.                     
 ;              Patt, the wild card file name pattern.                     
 ;   Calls nothing, returns a list of filenames, each with a path.         
 ; 
 (DEFUN LIME (cdir patt / delist filist delist nxdir nufils gnulst gnudi dnam
                                                            sport dirp nulist)
  (setq sport t)
  (setq filist (vl-directory-files cdir patt 1))
  (if sport (setq filist (acad_strlsort filist)))
  (setq filist (mapcar '(lambda (fnam) (strcat cdir fnam)) filist))
  (setq delist (vl-directory-files cdir nil -1))
  (if sport (setq delist (acad_strlsort delist)))
  (while (setq dirp (car delist))
         (setq delist (cdr delist))
         (if (not (member dirp '("." "..")))
             (setq gnulst (append gnulst (list dirp)))))
  (setq delist (mapcar '(lambda (dnam) (strcat cdir dnam "\\")) gnulst))
  (while (setq nxdir (car delist))
         (setq delist (cdr delist))
         (setq gnudi (vl-directory-files nxdir nil -1))
         (if sport (setq gnudi (acad_strlsort gnudi)))
         (setq nulist ())
         (while (setq dirp (car gnudi))
                (setq gnudi (cdr gnudi))
                (if (not (member dirp '("." "..")))
                    (setq nulist (append nulist (list dirp)))))
         (setq gnudi (mapcar '(lambda (dnam) (strcat nxdir dnam "\\")) nulist))
         (setq delist (append delist gnudi))
         (setq nufils (vl-directory-files nxdir patt 1))
         (if (and sport nufils) (setq nufils (acad_strlsort nufils)))
         (setq nufils (mapcar '(lambda (fnam) (strcat nxdir fnam)) nufils))
         (setq filist (append filist nufils)))
 filist)
 ; Ŀ
 ;   Lime end.                                                             
 ; 

 ; Ŀ
 ;   Lime.  Apparently a test program for (Lime).                          
 ; 
 (DEFUN C:LIME (/ cdir)
 ; Ŀ
 ;   Decide which directory the current drawing is in.                     
 ; 
  (setq cdir (getvar "dwgprefix"))
  (print (lime cdir "*.dwg"))
 (princ))

 ; Ŀ
 ;   Subroutine Liss: make a list of entity names into an ss.              
 ;   Arguments: Liz, a list of enames.                                     
 ;   Returns an ss.                                                        
 ; 
 (DEFUN LISS (liz / num ss enam)
  (setq num 0)
  (setq ss (ssadd))
  (while (setq enam (nth num liz))
         (ssadd enam ss)
         (setq num (1+ num)))
 ss)
 ; Ŀ
 ;   Subroutine Liss end.                                                  
 ; 

 ; Ŀ
 ;   Litof - make a drawing name list into a csv file.                     
 ;   Arguments: Filnam, a path and file name.                              
 ;              Liz, a list of file names.                                 
 ;   Calls Nothing.                                                        
 ;   Returns nothing.                                                      
 ; 
 (DEFUN LITOF (filnam liz / prefa fn num str)
 ; Ŀ
 ;   Add the current path and the extension .dwg to each drawing name.     
 ; 
  (setq prefa (getvar "dwgprefix"))
  (setq liz (mapcar '(lambda (fnam)
                      (strcat prefa fnam ".dwg"))
                      liz))
 ; Ŀ
 ;   Open the file, read and process each line.                            
 ; 
  (setq fn (open filnam "w"))
  (setq num 0)
  (while (setq str (nth num liz))
         (setq num (1+ num))
         (write-line str fn))
  (close fn)
 (princ))
 ; Ŀ
 ;   Litof end.                                                            
 ; 

 ; Ŀ
 ;   Subroutine Longa: find the longest attribute width at a given         
 ;   position in a block for an ss.                                        
 ;   Arguments: Ss, an ss of block insertions.                             
 ;              Poz, a zero based position in the block.                   
 ;                                                                         
 ;   Returns a width, possibly zero or nil.                                
 ; 
 (DEFUN LONGA (ss poz / num enam widmax widp)
  (setq num 0)
  (while (setq enam (ssname ss num))
         (setq num (1+ num))
         (setq widp (awp enam 0))
         (if (or (null widmax)
                 (and widp (> widp widmax)))
             (setq widmax widp)))
 widmax)
 ; Ŀ
 ;   Subroutine Longa end.                                                 
 ; 

 ; Ŀ
 ;   Subroutine Lowest - find the smallest leading number in a sublist     
 ;   of the list Nexlst which is the sole argument.                        
 ;   Revised 2007.10.02 - leading numbers can be ints, reals, or number    
 ;   strings.                                                              
 ; 
 (DEFUN LOWEST (nexlst / num minlst neth carna)
  (setq num 0)
  (setq minlst (list min))
  (while (setq neth (nth num nexlst))
         (setq carna (car neth))
         (if (= 'str (type carna)) (setq carna (read carna)))
         (if neth (setq minlst (append minlst (list carna))))
         (setq num (1+ num)))
  (eval minlst))
 ; Ŀ
 ;   Lowest end.                                                           
 ; 

 ; Ŀ
 ;   Mint - Convert seconds to hours, minutes and seconds.                 
 ; 
 (DEFUN MINT (sec / hours mins)
  (setq hours (itoa (fix (/ sec 3600))))
  (setq sec (rem sec 3600))
  (setq mins (itoa (fix (/ sec 60))))
  (setq sec (itoa (fix (rem sec 60))))
  (if (= (strlen hours) 1) (setq hours (strcat "0" hours)))
  (if (= (strlen mins) 1) (setq mins (strcat "0" mins)))
  (if (= (strlen sec) 1) (setq sec (strcat "0" sec)))
  (strcat hours ":" mins ":" sec))
 ; Ŀ
 ;   Mint end.                                                             
 ; 

 ; Ŀ
 ;   Subroutine Nopth - remove the path from a filename.                   
 ; 
 (DEFUN NOPTH (tt / pos)
  (setq pos (strlen tt))                          ; start at end of the string
  (while (< 0 pos)
          (if (or (= (substr tt pos 1) (chr 92))  ; if char = \
                  (= (substr tt pos 1) ":"))      ; if char = :
             (progn
                   (setq tt (substr tt (1+ pos))) ; then set tt to all after
                   (setq pos 1)))                 ;  and set pos to first
         (setq pos (1- pos)))                     ; set pos to previous
  tt)
 ; Ŀ
 ;   Nopth end.                                                            
 ; 

 ; Ŀ
 ;   Nyx - get a list of lisp routines to run.                             
 ;   Disallows any files which it can't find and any in the list Noron     
 ;   which shouldn't be run from a script.                                 
 ;   Returns a list of lisp names.                                         
 ; 
 (DEFUN NYX (/ noron num lispnm lnam lsplst foon)
  (setq noron (list "fang" "tch" "tctitle"))  ; stuff not to run from fang
  (setq num 1)
 ; Ŀ
 ;   Get the list of lisps to run.                                         
 ; 
  (while (or (/= "" (setq lispnm (getstring (strcat "Enter lisp name "
                                                    (itoa num) ": "))))
             (null lsplst))
         (cond ((member lispnm noron)
                (setq lnam (strcat (strcase (substr lispnm 1 1))
                                   (strcase (substr lispnm 2) t)))
                (prompt (strcat "Can't run " lnam " from a script.\n")))
               ((or (setq foon (findfile (strcat lispnm ".lsp")))
                    (setq foon (findfile (strcat lispnm ".fas"))))
                (grtext num lispnm)
;               (prompt (strcat foon "\n"))
                (setq num (1+ num))
                (grtext num " ")
                (grtext (1+ num) " ")
                (setq lsplst (append lsplst (list lispnm))))
               ((and (= lispnm "") (null lsplst))
                (prompt "Fang can't run on empty.\n"))
               (t
                (prompt "No such file found.\n"))))
 lsplst)
 ; Ŀ
 ;   Nyx end.                                                              
 ; 

 ; Ŀ
 ;   Subroutine Pthonl - remove everything but the path from a filename.   
 ; 
 (DEFUN PTHONL (tt / pos)
  (setq pos (strlen tt))                          ; start at end of the string
  (while (< 0 pos)
          (if (or (= (substr tt pos 1) (chr 92))  ; if char = \
                  (= (substr tt pos 1) ":"))      ; if char = :
             (progn
                   (setq tt (substr tt 1 pos))    ; then set tt to all before
                   (setq pos 1)))                 ;  and set pos to first
         (setq pos (1- pos)))                     ; set pos to previous
  tt)
 ; Ŀ
 ;   Pthonl end.                                                           
 ; 

 ; Ŀ
 ;   Sdia: alarm and exit if more than one drawing is open.                
 ;   Takes no arguments, returns nothing, exits if > 1 dwgs are open.      
 ; 
 (DEFUN SDIA ()
  (setq curdox (vla-get-documents (vlax-get-acad-object)))
  (setq dwgsop (vla-get-count curdox))
  (cond ((> dwgsop 6)
         (alert "Fang must be started with only one drawing open. ")
         (exit))
        ((> dwgsop 1)
         (setq namstr (strcat
                          "Fang must be started with only one drawing open.  "
                          "\n\nCurrent drawings:"))
         (vlax-for dwga curdox
                  (setq namstr (strcat namstr "\n" (vla-get-name dwga))))
         (alert namstr)
         (exit)))
 (princ))
 ; Ŀ
 ;   Sdia end.                                                             
 ; 

 ; Ŀ
 ;   Spit - returns the insertion point of the text entity whose data was  
 ;   passed as its sole argument.  Note that this is not necessarily the   
 ;   same as the 10 association code.                                      
 ; 
 (DEFUN SPIT (entt / xjust yjust)
  (setq xjust (cdr (assoc 72 entt)))
  (if (= (cdr (assoc 0 entt)) "ATTDEF")
      (setq yjust (cdr (assoc 74 entt)))
      (setq yjust (cdr (assoc 73 entt))))
  (if (or (/= xjust 0) (/= yjust 0))
      (cdr (assoc 11 entt))
      (cdr (assoc 10 entt))))
 ; Ŀ
 ;   Spit end.                                                             
 ; 

 ; Ŀ
 ;   Subroutine Splat - divide a text string into a list of substrings.    
 ;   Arguments: Sepchr, the field separator character.                     
 ;              Linn, the text string.                                     
 ;   Returns a list of field values, removes leading and trailing spaces.  
 ;                                                                         
 ;   Completely rewritten 11.19.2000.                                      
 ; 
 (DEFUN SPLAT (sepchr linn / pos len name1 strlst)
  (while (/= (strlen linn) 0)
 ; Ŀ
 ;   Find the first separator character, save everything before it into    
 ;   the Name1 variable, remove it from the start of the string Linn.      
 ; 
         (setq pos 1)
         (setq len (strlen linn))
         (while (and (/= (substr linn pos 1) sepchr)
                     (>= len pos))
                (setq pos (1+ pos)))
         (setq name1 (substr linn 1 (1- pos)))
         (setq linn (substr linn (1+ pos)))
 ; Ŀ
 ;   Remove spaces from the front and back of Name1.                       
 ; 
         (while (and (> (strlen name1) 0)
                     (= (substr name1 (setq len (strlen name1))) " "))
                (setq name1 (substr name1 1 (1- len))))
         (while (and (> (strlen name1) 0)
                     (= (substr name1 1 1) " "))
                (setq name1 (substr name1 2)))
 ; Ŀ
 ;   Add Name1 to the substring list Strlst.                               
 ; 
         (setq strlst (append strlst (list name1))))
 ; Ŀ
 ;   If the string contained no separator characters then Strlst will be   
 ;   nil, so return a list containing the original string.                 
 ; 
  (if (null strlst) (setq strlst (list linn)))
 strlst)
 ; Ŀ
 ;   Splat end.                                                            
 ; 

 ; Ŀ
 ;   Subroutine Stoker - write the script file Fiat_lux.scr.               
 ;   Arguments: Prefa,  the script file location.                          
 ;              Lsplst, the lisp list.                                     
 ;              Filist, the file name list.                                
 ;              Conf,   the end or quit string.  (Used?)                   
 ;              Eoq,    the End or Quit each drawing flag.                 
 ;   And two arguments to be written into the script file for Timer:       
 ;              Numlin, the total number of drawings.                      
 ;              Start,  the start time.                                    
 ;                                                                         
 ;   Calls Nopth, returns nothing.                                         
 ;                                                                         
 ;   Note that while a script file can span multiple drawings, a lisp      
 ;   function can not: the Cond code which writes the error file and       
 ;   opens the next drawing ends when the drawing is opened.               
 ;   The code to see if the correct drawing is open and run the lisp       
 ;   routines is a separate chunk of code.                                 
 ; 
 (DEFUN STOKER (prefa lsplst filist conf eoq numlin start / fbat numm var base
                                                      drawl errnam lispnm endp)
  (setq numm 1)
  (setq fbat (open (strcat prefa "fiat_lux.scr") "w"))
  (while (setq var (car filist))
         (setq filist (cdr filist))
 ; Ŀ
 ;   Replace backslashes in the path with double backslashes.              
 ; 
         (setq var (car (chug "\\" "\\\\" var)))
 ; Ŀ
 ;   Get the base part of the filename without the extension.              
 ; 
         (if (= (strcase (substr var (- (strlen var) 3))) ".DWG")
             (setq base (substr var 1 (- (strlen var) 4)))
             (setq base var))
 ; Ŀ
 ;   Make the lock file name.                                              
 ; 
         (setq drawl (strcat base ".dwl"))
 ; Ŀ
 ;   Make an error file name in case one is needed.                        
 ; 
         (setq errnam (strcat (pthonl var) "error.txt"))
 ; Ŀ
 ;   Write the timer/lap counter code.                                     
 ; 
         (write-line (strcat
               "(if (not timer) (load " (chr 34) "fang" (chr 34) "))") fbat)
         (write-line (strcat "(timer " (rtos start 2 2) " " (itoa numlin)
                                               " " (itoa numm) ")") fbat)
 ; Ŀ
 ;   Write the code to turn on single document interface mode.             
 ;   This will be added before each Open command in case something in      
 ;   the setup turns it back off.                                          
 ; 
         (write-line "(setvar \"sdi\" 1)" fbat)
 ; Ŀ
 ;   Increment the lap counter.                                            
 ; 
         (setq numm (1+ numm))
 ; Ŀ
 ;   Add the code to log an error if the file wasn't found.                
 ; 
(write-line (strcat "(cond ((null (findfile \"" var "\"))") fbat)
(write-line (strcat "       (setq fn (open \"" errnam "\" \"a\"))") fbat)
(write-line (strcat "       (write-line \"Unable to find " var ".\" fn)") fbat)
(write-line         "       (close fn))" fbat)
(write-line (strcat "      ((and (findfile \"" drawl "\")") fbat)
(write-line (strcat "            (/= (strcase (strcat (getvar \"dwgprefix\")"
                    " (getvar \"dwgname\")))") fbat)
(write-line (strcat "                \"" (strcase var) "\"))") fbat)
(write-line (strcat "       (setq fn (open \"" errnam "\" \"a\"))") fbat)
(write-line (strcat "       (write-line \"File " var " is locked.\" fn)") fbat)
(write-line         "       (close fn))" fbat)
(write-line         "      (t" fbat)
(write-line         "       (command \"open\")" fbat)
(write-line         "       (if (> (rem (getvar \"dbmod\") 32) 0)" fbat)
(if eoq (write-line         "           (command \"N\")" fbat)
        (write-line         "           (command \"Y\")" fbat))
(write-line         "           (princ))" fbat)
(write-line (strcat "       (command \"" var "\")))") fbat)
 ; Ŀ
 ;   Set the position in the lisp list variable.                           
 ; 
         (setq num 0)
 ; Ŀ
 ;   Check to see if are in the correct drawing before changing anything.  
 ; 
         (write-line (strcat "(if (= (strcase (getvar \"dwgname\")) \""
                                     (strcase (nopth var)) "\")") fbat)
         (write-line "    (progn" fbat)
 ; Ŀ
 ;   Write the name of each lisp in the lsplst to the file - load and      
 ;   then run it.                                                          
 ; 
         (while (setq lispnm (nth num lsplst))
                (setq num (1+ num))
                (if (= num (length lsplst))
                    (setq endp "))")
                    (setq endp ""))
                (write-line (strcat "         (load " (chr 34) lispnm
                                                      (chr 34) ")") fbat)
                (write-line (strcat "         (C:" lispnm ")" endp) fbat)))
 ; Ŀ
 ;   End of While Filenames loop.  Write the code to set SDI back to       
 ;   what it was, which just happens to be what it is.                     
 ; 
  (setq sdi (getvar "sdi"))
  (write-line (strcat "(setvar \"sdi\" " (itoa sdi) ")") fbat)
 ; Ŀ
 ;   Add the commands to delete the script and either End or Quit.         
 ; 
  (write-line (strcat "shell del " prefa "fiat_lux.scr") fbat)
  (if eoq
     (write-line "END" fbat)
     (progn
          (write-line "Quit" fbat)
          (write-line "Y" fbat)))                    ; really discard etc.?
  (close fbat)
 (princ))
 ; Ŀ
 ;   Stoker end.                                                           
 ; 

 ; Ŀ
 ;   Timer - Script file timer.                                            
 ;   Copyright 1994 by Rocket Software Ltd.                                
 ;   Software to help alleviate the tedium of mindless labour.             
 ; 
 (DEFUN TIMER (t1 totnum done / s t2 elapse timea timlft)
 ; Ŀ
 ;   T1 = start time, Totnum = total drawings, Done = drawings finished    
 ; 
  (setq s (getvar "date"))
  (setq t2 (* 86400.0 (- s (fix s))))
  (setq elapse (- t2 t1))
 ; Ŀ
 ;   Write the lap counter before Done is decremented.                     
 ; 
  (grtext 1 " ")
  (grtext 2 (strcat (itoa done) "/" (itoa totnum)))
  (grtext 3 " ")
  (setq done (1- done))   ; decrement - timer called before curr. dwg processed
  (if (> done 0)
      (progn
           (setq timea (/ elapse (float done)))
           (setq timlft (* timea (- totnum done)))
           (grtext 4 " ")
           (grtext 5 "Time")
           (grtext 6 "--------")
           (grtext 7 "Elapsed:")
           (grtext 8 (setq elap (mint elapse)))
           (grtext 9 "--------")
           (grtext 10 "Left:")
           (grtext 11 (setq tleft (mint timlft)))
           (grtext 12 "--------")
           (grtext 13 " ")
           (grtext 14 " ")
           (grtext 15 " ")
           (setvar "modemacro" (strcat (itoa done) "/" (itoa totnum)
                                       "  Time Elapsed: " elap
                                       "  Time Left: " tleft))))
 (princ))
 ; Ŀ
 ;   Timer end.                                                            
 ; 

 ; Ŀ
 ;   Twid - find the width of a hypothetical text string.                  
 ;   Takes one argument, the text entity data list, returns a length.      
 ; 
 (DEFUN TWID (entt / tblist cc dd bwidth)
  (setq tblist (textbox entt))
  (setq cc (car tblist))                    ; ll offset from 10 of text
  (setq dd (cadr tblist))                   ; ur offset from 10 of text
  (setq bwidth (- (car dd) (car cc))))
 ; Ŀ
 ;   Twid end.                                                             
 ; 

 ; Ŀ
 ;   Subroutine Vtol: returns a list of enames ordered entity position.    
 ;   Arguments: Ss, a selection set of entities to order.                  
 ;              Dir, a direction - if this is either "X" or "Y" then the   
 ;                   entities are assumed to be arrayed in that direction, 
 ;                   if anything else then the routine uses the direction  
 ;                   in which they are most spread out.                    
 ;              Insa, if T and the entity is text or an attribute then     
 ;                    sort based on the insertion point rather than the    
 ;                    ten point.                                           
 ;                                                                         
 ;   This is the latest version: 2006.10.05, which sorts text by           
 ;   insertion point rather than ten point and in which setting the        
 ;   direction to nil doesn't cause a crash.                               
 ;   Also it works with attdefs as well as text.                           
 ;   It should replace all other uses of Vtol and Stol.                    
 ;                                                                         
 ;   Revamped 2009.07.28 to use Apply rather than Eval Cons 'Max List etc. 
 ;   This is less elegant but removes the 256 entity limitation.           
 ;   Also added the ability to sort by either ten point or insertion.      
 ;                                                                         
 ; 
 (DEFUN VTOL (ss dir insa / xposnam yposnam numm ent entt ten xpos ypos xx yy
                            pn maxx minx maxy miny xdif ydif poslst posnam
                                                       direct pos lastt order)
  (setq xposnam ())                      ; initialize (xpos & name list) list
  (setq yposnam ())                      ; initialize (ypos & name list) list
 ; Ŀ
 ;   Now see if the entities are arranged horizontally or vertically.      
 ; 
  (setq numm 0)                             ; start at the ss beginning again
  (while (setq ent (ssname ss numm))
         (setq entt (entget ent))
         (if (and insa (member (cdr (assoc 0 entt)) '("TEXT" "ATTDEF")))
             (setq ten (spit entt))
             (setq ten (cdr (assoc 10 entt))))
         (setq xpos (car ten))
         (setq ypos (cadr ten))
         (setq xx (append xx (list xpos)))  ; add x insert to list
         (setq yy (append yy (list ypos)))  ; and y to y list
 ; Ŀ
 ;   Also make the position and name list.  Have to make one for the X     
 ;   values and one for the Ys and use the appropriate one later.          
 ; 
         (setq pn (cons xpos ent))
         (setq xposnam (append xposnam (list pn)))
         (setq pn (cons ypos ent))
         (setq yposnam (append yposnam (list pn)))
         (setq numm (1+ numm)))             ; next entity
 ; Ŀ
 ;   Now evaluate the four lists.  The result will be the max and min      
 ;   values for the X and Y lists.                                         
 ; 
  (setq maxx (apply 'max xx))
  (setq minx (apply 'min xx))
  (setq maxy (apply 'max yy))
  (setq miny (apply 'min yy))
  (setq xdif (- maxx minx))
  (setq ydif (- maxy miny))
 ; Ŀ
 ;   Set direction variables to match whichever direction was given in     
 ;   the argument, if it was nil then deduce a direction.                  
 ; 
  (cond ((and (= (type dir) 'STR)
              (= (strcase dir) "X"))
          (setq poslst xx)                ; positions from X coord list
          (setq posnam xposnam)           ; position & ename list with X coord
          (setq direct 'min))             ; edit from smallest to largest X
        ((and (= (type dir) 'STR)
              (= (strcase dir) "Y"))
         (setq poslst yy)
         (setq posnam yposnam)
         (setq direct 'max))
        (T
 ; Ŀ
 ;   The default case: figure it out yourself.                             
 ;   Set vert to T if vertical, nil if horizontal.                         
 ;   If not sure, assume vertical.                                         
 ;   Could set strip to Quit and thus do so...                             
 ; 
         (cond ((> xdif ydif)             ; if (Xmax - Xmin) > (Ymax - Ymin)
                (setq poslst xx)          ; positions from X coord list
                (setq posnam xposnam)     ; position & ename list with X coord
                (setq direct 'min))       ; edit from smallest to largest X
               ((< xdif ydif)
                (setq poslst yy)
                (setq posnam yposnam)
                (setq direct 'max))
               (T                         ; if not sure then call it vertical
                (setq poslst yy)
                (setq posnam yposnam)
                (setq direct 'max)))))
 ; Ŀ
 ;   Now make the list of enames in order by increasing X or decreasing Y  
 ;   coordinate depending on whether the array is horizontal or vertical.  
 ;   Already Have Posnam: a list of (list position ename).                 
 ;   Using the original list of either x or y positions, get the first or  
 ;   last as appropriate, extract the ename from Posnam using              
 ;   (cdr (assoc (largest Y or smallest X) posnam))                        
 ;   and append the ename to the end of the enames in order list: Order.   
 ;   Then remove that position from the position list.                     
 ; 
  (while (> (length poslst) 0)
 ; Ŀ
 ;   Get the largest Y or smallest X value in the position list.           
 ; 
         (setq maxx (apply direct poslst))
 ; Ŀ
 ;   Having found Maxx, want to remove that value from poslst.             
 ;   Get the list from Maxx on, and the position of Maxx within the list.  
 ; 
         (setq pos (- (length poslst)
                      (length (setq lastt (member maxx poslst)))))
 ; Ŀ
 ;   Get the list after maxx.                                              
 ; 
         (setq lastt (cdr lastt))
 ; Ŀ
 ;   And add the list members before maxx.                                 
 ;   One could use (cdr (member (reverse poslist))) but if there were two  
 ;   values the same in the list this would result in a longer rather      
 ;   than a shorter poslist.                                               
 ; 
         (setq pos (1- pos))     ; subtract one: nth is zero based
         (while (>= pos 0)
                (setq lastt (append (list (nth pos poslst)) lastt))
                (setq pos (1- pos)))
         (setq poslst lastt)      ; poslst becomes lastt
 ; Ŀ
 ;   Now get the matching ename from posnam and add it to the end of the   
 ;   order list.                                                           
 ; 
         (setq order (append order (list (cdr (assoc maxx posnam)))))
 ; Ŀ
 ;   If there are two entities with the same position then assoc will      
 ;   always return the first one.  Must delete the first one each time -   
 ;   subst (nil) for it.                                                   
 ; 
         (setq posnam (subst (list nil) (assoc maxx posnam) posnam)))
 order)
 ; Ŀ
 ;   Vtol end.                                                             
 ; 

 ; Ŀ
 ;   Subroutine Fang.  The main structure.                                 
 ;   Argument: Deta - if nil then only ask to edit the drawing name file,  
 ;                    for lisp names, and whether to save or quit.         
 ;                    If T then offer a lot more options.                  
 ;   Called by either Fang or Renfield, calls everything else.             
 ;   Returns nothing.                                                      
 ; 
 (DEFUN FANG (deta / usep filp filnam flist pat pref prefa num fn namm filist
                                          numlin lsplst eoq conf s start insp)
  (setvar "cmdecho" 0)
 ; Ŀ
 ;   Display a warning and exit if more than one drawing is open.          
 ;   This could be modified so that it only cared if the user tried to     
 ;   run the script file, but that isn't very likely.                      
 ;   Apparently we have to check - pre-2006 versions don't support the     
 ;   vlax-etc function call.                                               
 ; 
 (if (boundp 'vlax-get-acad-object)
     (sdia))
 ; Ŀ
 ;   Either A. Get a file listing or B. Use an existing directory file.    
 ;   Ask if want to edit the file.  If so:                                 
 ;   Either A. Dump into a file for editing or B. Edit the existing file.  
 ;   If not then read the file into a list.                                
 ;   Get lisp names, write the script, run it.                             
 ;                                                                         
 ;   First either get a filename or make a new filename list file.         
 ; 
  (if deta
      (progn
           (initget 0 "Use Saved File Make New File Listing Blocks")
           (Setq usep (getkword
                  "Use Saved file/select a Block/<Make New File Listing>: "))))
 ; Ŀ
 ;   Cond 1/3: If making a new file:                                       
 ;   (Which will be the case if Deta is nil and we are assuming defaults.) 
 ; 
  (cond ((or (null usep) (member usep '("Make" "New" "File" "Listing")))
 ; Ŀ
 ;   Ask what to call the drawing list file.                               
 ; 
         (if (null filnam) (setq filnam "Files.dat"))
         (if deta
            (setq filp (getstring (strcat "File list filename <"
                                           filnam ">: "))))
         (if (and filp (/= filp "")) (setq filnam filp))
 ; Ŀ
 ;   Make sure the names file is in the dir with the current drawing.      
 ; 
         (setq filnam (strcat (getvar "dwgprefix") filnam))
 ; Ŀ
 ;   See if should process subdirectories.                                 
 ; 
         (if deta
            (progn
                 (initget 0 "Process Subdirectories Yes No")
                 (setq subb (getkword "Process Subdirectories - Yes/<No>: "))))
         (if (or (null subb) (= subb "No"))
 ; Ŀ
 ;   Don't do subdirectories: call Dirge to get the list of files and      
 ;   the directory path.                                                   
 ; 
             (progn
                  (setq flist (dirge deta))
                  (setq prefa (car flist))
                  (setq flist (cadr flist)))
 ; Ŀ
 ;   Do.                                                                   
 ; 
             (progn
                  (if (or (null deta)
                          (= "" (setq pat (getstring "Pattern <*.dwg>: "))))
                      (setq pat "*.dwg"))
                  (setq pref (getvar "dwgprefix"))
                  (if (or (null deta)
                          (= "" (setq prefa (getstring (strcat "Directory <"
                                                                pref ">: ")))))
                      (setq prefa pref))
                  (setq flist (lime prefa pat))))
 ; Ŀ
 ;   Dump them into a text file.                                           
 ; 
         (setq num 0)
         (setq fn (open filnam "w"))
         (while (setq namm (nth num flist))
                (write-line namm fn)
                (setq num (1+ num)))
         (close fn)
 ; Ŀ
 ;   Don't go on until the file has been created.                          
 ; 
         (setq num 0)
         (while (not (findfile filnam))
                (if (= 12 (setq num (1+ num)))
                    (progn
                         (alert (strcat "System Error:\nCan't find "
                                         filnam "."))
                         (exit)))
                (command "delay" 250)))
 ; Ŀ
 ;   Cond 2/3: If on the other hand we are not making a new directory list 
 ;   file, ask for an existing one.                                        
 ; 
        ((member usep '("Use" "Saved"))
         (setq filnam (getfiled "Directory list File" "" "" 0)))
 ; Ŀ
 ;   Cond 3/3: extract file names from index blocks in the drawing.        
 ;   The drawing name is taken to be the first attribute.                  
 ; 
        ((member usep '("Blocks"))
         (setq filnam (strcat (getvar "dwgprefix") "files.dat"))
         (colm)))
 ; Ŀ
 ;   If using an existing drawing name file then Dirge won't have been     
 ;   called so we won't have a path Prefa to use as a location for the     
 ;   script file Fiat_Lux.scr.  A premade drawing list file may contain    
 ;   files in different locations, so we have no way of knowing where to   
 ;   put Fiat_lux, and it seems like overkill to ask, so put it in the     
 ;   current directory.                                                    
 ; 
  (if (null prefa) (setq prefa (getvar "dwgprefix")))
 ; Ŀ
 ;   Should now have a file name and a matching data file.                 
 ;   See if the user wants to edit the file.                               
 ; 
  (initget 0 "Yes No")
  (Setq usep (getkword "Stop to edit drawing name file? <N>: "))
  (if (= usep "Yes")
      (progn
           (command "not" filnam)
           (command "delay" 250)))
 ; Ŀ
 ;   If the file is a .csv or .cdf then must call Ceecee to read it into   
 ;   a list of file names, Ceecee does a fair amount of rehashing.         
 ;   Deep handles all other data files.                                    
 ; 
  (if (member (strcase (substr filnam (- (strlen filnam) 3))) '(".CSV" ".CDF"))
      (setq filist (ceecee filnam))
 ; Ŀ
 ;   Call Deep to read the file (edited or not) back into a list,          
 ;   deleting empty lines and comments.                                    
 ;   Might want to add the option to dispense with the file entirely.      
 ;   Or use a ddedit box and make saving and reading options.              
 ; 
      (setq filist (deep filnam)))
 ; Ŀ
 ;   Find the length of the filename list (i.e. the number of files.)      
 ; 
  (setq numlin (length filist))
 ; Ŀ
 ;   Call Nyx to get the list of lisps to run.                             
 ; 
  (setq lsplst (nyx))
 ; Ŀ
 ;   Ask whether to Quit or End.                                           
 ; 
  (setq eoq (getstring "End or Quit after each drawing <End>: "))
  (if (or (= eoq "") (= eoq "E") (= eoq "END"))
      (setq eoq T)
      (setq eoq ()))
 ; Ŀ
 ;   Initialise the quit or end confirmation string for the script file.   
 ;   ... this doesn't seem to be used anywhere - stoker makes its own.     
 ; 
 ; Ŀ
 ;   Dbmod is a bitcoded value indicating in what way the drawing has      
 ;   changed, using the sum of the following values:                       
 ; Ĵ
 ;   1     Object database modified.                                      
 ;   2     Aparently not used.                                            
 ;   4     Database variable modified.                                    
 ;   8     Window modified.                                               
 ;   16    View modified.                                                 
 ;   32    Field modified.                                                
 ; Ĵ
 ;   Dbmod is reset to 0 when you save the drawing.                        
 ;   The point here is that if Dbmod isn't either 0 or 32 then Acad will   
 ;   ask whether to save the drawing before it is closed to open another.  
 ; 
  (setq conf (strcat "        (if (> (rem (getvar \"dbmod\") 32) 0)"
                     "\n            (command \"" (if eoq "N" "Y") "\")"
                     "\n            (princ))"))
 ; Ŀ
 ;   Initialise start time variable for Timer.                             
 ; 
  (setq s (getvar "date"))
  (setq start (* 86400.0 (- s (fix s))))          ; start time for timer
 ; Ŀ
 ;   Write the script file.                                                
 ; 
  (stoker prefa lsplst filist conf eoq numlin start)
 ; Ŀ
 ;   Offer to run Fiat_lux, unless deta is nil in which case do it and     
 ;   don't ask, end.                                                       
 ; 
  (if deta
      (progn
           (initget 0 "Yes No")
           (Setq insp (getkword "Start the script? <Y>: "))))
  (if (or (null insp) (= insp "Yes"))
      (command "script" (strcat prefa "Fiat_lux")))
 (princ))
 ; Ŀ
 ;   Subroutine Fang end.                                                  
 ; 

 ; Ŀ
 ;   Fangs - ask for everything.                                           
 ; 
 (DEFUN C:FANGS ()
  (fang t)
 (princ))

 ; Ŀ
 ;   Fang - the quick version.                                             
 ; 
 (DEFUN C:FANG ()
  (fang nil)
 (princ))


